#ifndef __CMidiSequence__
#define __CMidiSequence__

//	===========================================================================

#include <Collections/TCountedPointerArray.hpp>
#include <Basics/CSystemString.hpp>
#include "CMidiTrack.hpp"
#include "../Music/CTimeSignature.hpp"
using Exponent::Collections::TCountedPointerArray;
using Exponent::Basics::CSystemString;
using Exponent::Midi::CMidiTrack;
using Exponent::Music::CTimeSignature;

//	===========================================================================

namespace Exponent
{
	namespace Midi
	{
		/**
		 * @class CMidiSequence CMidiSequence.hpp
		 * @brief Provides an extensible array of midi events. As this can cause dynamic allocation this class is not intended for real time usage when writing, reading is fine dynamically\n
		 *
		 * It should be noted that midi sequences store a bunch of tracks. Each track contains a collection of CMidiEvents. In this case the delta time for each event is a sample from a notional\n
		 * zero position, that is essentially the start of the song.
		 *
		 * @see CMidiTrack
		 * @see CMidiFile
		 *
		 * @date 23/08/2004
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * @note Documented online at <A HREF="http://www.borg.com/~jglatt/tech/midifile.htm">http://www.borg.com/~jglatt/tech/midifile.htm</A>
		 *
		 * $Id: CMidiSequence.hpp,v 1.2 2007/02/08 21:08:09 paul Exp $
		 */
		class CMidiSequence : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			 * Construction
			 * @param sampleRate The sample rate that we are playing back at
			 */
			CMidiSequence(const double sampleRate);

			/**
			 * Destruction
			 */
			virtual ~CMidiSequence();

//	===========================================================================

			/**
			 * Add a track
			 * @param track The midi event track
			 */
			void addTrack(CMidiTrack *track);

			/**
			 * Set the sample rate. Note that this will cause the recomputation of all the events, which may take a long time. Be sure that you have got the time to perform this operation before you do it
			 * @param sampleRate The new sample rate
			 */
			void setSampleRate(const double sampleRate);

			/**
			 * Set the tempo
			 * @param tempo The tempo in bpm
			 */
			void setTempo(const double tempo);

			/**
			 * Set the time signature
			 * @param timeSignature The time signature
			 */
			void setTimeSignature(const CTimeSignature &timeSignature);

//	===========================================================================

			/**
			 * Get the sample rate
			 * @retval double The sample rate
			 */
			double getSampleRate() const { return m_sampleRate; }

			/**
			 * Get the tempo
			 * @retval double The tempo
			 */
			double getTempo() const { return m_tempo; }

			/**
			 * Get the time signature
			 * @retval const CTimeSignature& The time signature
			 */
			const CTimeSignature &getTimeSignature() const { return m_timeSignature; }

			/**
			 * Get the number of tracks
			 * @retval long The total number of tracks
			 */
			long getNumberOfTracks() const { return m_tracks.getArraySize(); }

			/**
			 * Get track at index
			 * @param index The index of the track you desire, must be >= 0 && < getNumberOfTracks
			 * @retval const CMidiTrack* The midi track or null on erorr
			 */
			const CMidiTrack *getTrackAtIndex(const long index) const;

			/**
			 * Get the mutable track at index
			 * @param index The index of the track you desire, must be >= 0 && < getNumberOfTracks
			 * @retval CMidiTrack* The midi track or null on erorr
			 */
			CMidiTrack *getMutableTrackAtIndex(const long index);

//	===========================================================================

			/**
			 * Compact The list. Remove any dead tracks and then removes unused memory portions 
			 */
			void compact();

//	===========================================================================

		protected:

//	===========================================================================
		
			TCountedPointerArray<CMidiTrack> m_tracks;			/**< Tracks that we are dealing with */
			double m_tempo;										/**< BPM of the midi sequence */
			CTimeSignature m_timeSignature;						/**< The time signature */
			double m_sampleRate;								/**< SAmple rate of the system */
		};
	}
}
#endif	// End of CMidiSequence.hpp